#!/usr/bin/perl
# 
# Copyright (c) 2012, Oracle and/or its affiliates. All rights reserved. 
#
#  NAME           appcluster.pl
#
#  DESCRIPTION    Updates the configuration of Oracle Clusterware to be tailored
#                 for non-database applications by decreasing the memory 
#                 footprint and relaxing node synchronization requirements to be
#                 better suitable for non-database, yet business critical applications.
#
#                 This script needs to be run on all nodes of the cluster. The
#                 changes are not reversible, except by de-configuring and 
#                 re-configuring the cluster.
# 
#  MODIFIED      (MM/DD/YY)
#  dpham          10/30/12 - Fix bug 14828977
#  dpham          09/19/12 - Creation   
# 

package appcluster;

use strict;
use English;
use File::Basename;
use File::Spec::Functions;

BEGIN 
{
  # Add the directory of this file to the search path
  push @INC, dirname($PROGRAM_NAME);
}

# root script modules
use crsutils;
use s_crsutils;

my $ORACLE_HOME;
my ($SRVCTL, $CRSCTL);
my $keepcss = FALSE;
my $success = TRUE;

#*******************************************************************************
# Function: validate options
#*******************************************************************************
sub validateArgs
{
  my @argv = @_;
  if (scalar(@argv) == 0) #empty args
  {
    return TRUE;
  }

  my $option = $argv[0];

  # check for help option
  if ($option eq '--help' || $option eq '-h')
  {
    print_error(3500);
    print_error(3501, 'priority', 'misscount', 'disktimeout');
    return FALSE;
  }

  if (scalar(@argv) > 1)
  {
    $argv[1] =~ s/\-//g; #strip off --
    print_error(3000, $argv[1]);
    return FALSE;
  }
  elsif ($option ne '--keepcss')
  {
    $option =~ s/\-//g; #strip off --
    print_error(3000, $option);
    return FALSE;
  }

  if ($option eq '--keepcss')
  {
    $keepcss = TRUE;
  }

  return TRUE;
}

#*******************************************************************************
# Function: remove OC4J
#*******************************************************************************
sub removeOC4J
{
  trace("Removing OC4J....");

  my @cmd = ();
  my @out = ();
  my $rc;

  # stop ocj4
  @cmd = ($SRVCTL, 'stop', 'oc4j', '-f');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to stop oc4j, out=@out\n");
    print "@out\n";
  }

  # remove cvu
  @cmd = ($SRVCTL, 'remove', 'oc4j', '-f');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to remove oc4j, out=@out\n");
    print "@out\n";
  }
}

#*******************************************************************************
# Function: remove CVU
#*******************************************************************************
sub removeCVU
{
  trace("Removing CVU....");

  my @cmd = ();
  my @out = ();
  my $rc;

  # stop cvu
  @cmd = ($SRVCTL, 'stop', 'cvu', '-f');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to stop cvu, out=@out");
    print "@out\n";
  }

  # remove cvu
  @cmd = ($SRVCTL, 'remove', 'cvu', '-f');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to remove cvu, out=@out");
    print "@out\n";
  }
}

#*******************************************************************************
# Function: remove scan
#*******************************************************************************
sub removeScan
{
  trace("Removing scan....");

  my @cmd = ();
  my @out = ();
  my $rc;

  # stop/remove scan listener
  @cmd = ($SRVCTL, 'stop', 'scan_listener', '-f');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to stop scan listener, out=@out");
    print "@out\n";
  }
   
  @cmd  = ($SRVCTL, 'remove', 'scan_listener', '-y', '-f');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to remove scan listener, out=@out");
    print "@out\n";
  }
   
  # stop/remove scan vip
  @cmd = ($SRVCTL, 'stop', 'scan', '-f');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to stop scan, out=@out");
    print "@out\n";
  }
 
  # remove scan vip
  @cmd = ($SRVCTL, 'remove', 'scan', '-y', '-f');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to remove scan, out=@out");
    print "@out\n";
  }
}

#*******************************************************************************
# Function: remove nodeapps
#*******************************************************************************
sub removeNodeapps
{
  trace("remove nodeapps");
  my @cmd = ();
  my @out = ();
  my $rc;

  # stop nodeapps
  @cmd = ($SRVCTL, 'stop', 'nodeapps', '-f');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to stop nodeapps, out=@out");
    print "@out\n";
  }

  # remove nodeapps
  @cmd = ($SRVCTL, 'remove', 'nodeapps', '-y', '-f');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to remove nodeapps, out=@out");
    print "@out\n";
  }
}

#*******************************************************************************
# Function: Set css priority on other nodes
#*******************************************************************************
sub setCSSPriority
{
  trace("change css priority");

  # get css priority
  my @cmd = ($CRSCTL, 'get', 'css', 'priority');
  my @out = system_cmd_capture(@cmd);
  my $rc  = shift @out;
  trace("output=@out");

  # check if priority already set to 3
  if (scalar(grep(/^CRS-4269/, @out)) && scalar(grep(/ 3/, @out)))
  {
    print_error(3002);
    return;
  }

  # set css priority
  @cmd = ($CRSCTL, 'set', 'css', 'priority', '3');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;

  if ($rc != 0)
  {
    $success = FALSE;
    trace("failed to set css priority");
    print "@out\n";
  }
  else
  {
    print_error(3512);
  }

  return;
}

#*******************************************************************************
# MAIN SCRIPT BODY
#*******************************************************************************

my $PARAM_FILE = catfile(dirname($0), 'crsconfig_params');
my $osdfile = catfile(dirname($0), 's_crsconfig_defs');
crsutils->new(SIHA            => FALSE,
              paramfile       => $PARAM_FILE,
              osdfile         => $osdfile,
              crscfg_trace    => TRUE
             );

# set SRVCTL
$CRSCTL = catfile($CFG->params('ORACLE_HOME'), 'bin', 'crsctl');
$SRVCTL = catfile($CFG->params('ORACLE_HOME'), 'bin', 'srvctl');

# check for super user
if (!check_SuperUser())
{
  exit 1;
}

# validate options
if (!validateArgs(@ARGV))
{
  exit;
}

print_error(3510); #ask user to confirm
my $continue = <STDIN>;
chomp($continue);
if ($continue ne 'yes' and $continue ne 'YES')
{
  exit 1;
}

my @cmd = ();
my @out = ();
my $rc;
my ($ipv4Subnet, $ipv4Type, $ipv6Subnet);

# exit if it's SIHA
my $OCRCONFIG = $CFG->params('OCRCONFIG');
my @out = read_file ($OCRCONFIG);
if (scalar(grep(/^local_only=TRUE/, @out)))
{
  print_error(3003);
  exit 1;
}

# check if clusterware already trimmed
my $trimCluster = FALSE;
@cmd = ($CRSCTL, 'query', 'crs', 'skinny');
@out = system_cmd_capture(@cmd);
$rc  = shift @out;

if ($rc == 0)
{
  if (scalar(grep(/CRS-4562/i, @out)))
  {
    $trimCluster = TRUE;
  }
  else
  {
    # check if CRSSKINNY key
    if (scalar(grep(/^CRS-4565/, @out)) &&
        scalar(grep(/SYSTEM.CRSSKINNY/, @out)) &&
        scalar(grep(/ 0/, @out)))
    {
      $trimCluster = TRUE;
    }
  }
}
else
{
  trace("query crs skinny failed out=@out rc=$rc");
  print "@out\n";
  exit 1;
}


if (!$trimCluster)
{
  # if clusterware has already configured for non-database applications,
  # we need to set css priority on the other nodes
  if (!$keepcss)
  {
    setCSSPriority();
  }
  else
  {
    print_error(3002);
  }

  exit 1;
}

# configure non-database app
print_error(3511);

# get subnet/netmask/if
@cmd = ($SRVCTL, 'config', 'nodeapps');
@out = system_cmd_capture(@cmd);
$rc  = shift @out;

if ($rc == 0)
{
  # get Subnet IPv4
  my $subnetIPv4Txt = 'Subnet IPv4: ';
  my @network = grep(/$subnetIPv4Txt/, @out);
  if (scalar(@network))
  {
    trace("network=$network[0]");
    my $key;
    ($key, $ipv4Subnet) = split(/$subnetIPv4Txt/, $network[0]);
    ($ipv4Subnet, $ipv4Type) = split(/, /, $ipv4Subnet);
    trace("IPv4=$ipv4Subnet type=$ipv4Type");
  }

  # get Subnet IPv6
  my $subnetIPv6Txt = 'Subnet IPv6: ';
  my @network = grep(/$subnetIPv6Txt/, @out);
  if (scalar(@network))
  {
    trace("network=$network[0]");
    my ($key, $dummy);
    ($key, $ipv6Subnet) = split(/$subnetIPv6Txt/, $network[0]);
    ($ipv6Subnet, $dummy) = split(/, /, $ipv6Subnet);
    trace("IPv6=$ipv6Subnet ");
  }
}
else
{
  trace("srvctl config nodeapps failed out=@out rc=$rc");
  print_error(3004);
  print "@out\n";
  exit 1;
}

# remove CVU
removeCVU();

# remove OC4J
removeOC4J();

# remove SCAN
removeScan();

# remove nodeapps
removeNodeapps();

# add network IPv4
@cmd = ($SRVCTL, 'add', 'network', '-S', $ipv4Subnet, '-w', $ipv4Type);
@out = system_cmd_capture(@cmd);
$rc  = shift @out;
if ($rc != 0)
{
  trace("failed to add network out=@out rc=$rc");
  print "@out\n";
  exit 1;
}

# add network IPv6
if ($ipv6Subnet)
{
  @cmd = ($SRVCTL, 'add', 'network', '-S', $ipv6Subnet, '-w', 'AUTOCONFIG');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    trace("failed to add network out=@out rc=$rc");
    print "@out\n";
    exit 1;
  }
}

if (!$keepcss)
{
  my $rc;

  # set css priority
  @cmd = ($CRSCTL, 'set', 'css', 'priority', '3');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    $success = FALSE;
    trace("failed to set css priority");
    print "@out\n";
  }

  # set css misscount
  @cmd = ($CRSCTL, 'set', 'css', 'misscount', '90');
  @out = system_cmd_capture(@cmd);
  $rc  = shift @out;
  if ($rc != 0)
  {
    $success = FALSE;
    trace("failed to set css misscount");
    print "@out\n";
  }
}

# set skinny ocr key to 1
@cmd = ($CRSCTL, 'set', 'crs', 'skinny', '1');
@out = system_cmd_capture(@cmd);
$rc  = shift @out;

if ($rc != 0)
{
  $success = FALSE;
  trace("failed to set crs skinny");
  print "@out\n";
}

if ($success)
{
  print_error(3512);
}
else
{
  print_error(3513);
}


0;
